package panneaux;

import java.awt.AlphaComposite;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.RenderingHints;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;

import javax.swing.JPanel;

/**
 * @author Alexis_Deprez
 * JPanel deplacable pouvant recevoir une image de fond dont l'opacite peut changer
 */
public class PanelImage extends JPanel implements MouseListener, MouseMotionListener {
	private static final long serialVersionUID = 1L;
	private Image image;
	private int x,y;
	private float opacite=1;
	private int largeur, longueur;
	
	/**
	 * Constructeur vide
	 */
	public PanelImage() {}
	
	/**
	 * Cree une instance de PanelImage avec une image en fond
	 * @param image l'image de fond
	 */
	public PanelImage(Image image) {
		setImage(image);
	}
	
	/**
	 * Cree une instance de PanelImage avec une image en fond, de longueur et de largeur specifiee
	 * @param image l'image de fond
	 * @param largeur la largeur de cette image
	 * @param longueur la longueur de cette image
	 */
	public PanelImage(Image image, int largeur, int longueur) {
		setImage(image);
		setTaille(largeur, longueur);
	}
	
	/**
	 * Cree une instance de PanelImage avec une image en fond, de longueur et de largeur specifiee
	 * @param image l'image de fond
	 * @param taille les dimensions de cette image
	 */
	public PanelImage(Image image, Dimension taille) {
		setImage(image);
		setTaille(taille);
	}
	
	/**
	 * Cree une instance de PanelImage deplacable ou non avec une image en fond
	 * @param image l'image de fond
	 * @param deplacable si true, le PanelImage est deplacable
	 */
	public PanelImage(Image image, boolean deplacable) {
		setImage(image);
		if(deplacable) {
			addMouseListener(this);
			addMouseMotionListener(this);
		}
	}
	
	/**
	 * Cree une instance de PanelImage deplacable ou non avec une image en fond, de longueur et de largeur specifiee
	 * @param image l'image de fond
	 * @param largeur la largeur de cette image
	 * @param longueur la longueur de cette image
	 * @param deplacable si true, le PanelImage est deplacable
	 */
	public PanelImage(Image image, int largeur, int longueur, boolean deplacable) {
		setImage(image);
		setTaille(largeur, longueur);
		if(deplacable) {
			addMouseListener(this);
			addMouseMotionListener(this);
		}
	}
	
	/**
	 * Cree une instance de PanelImage deplacable ou non avec une image en fond, de longueur et de largeur specifiee
	 * @param image l'image de fond
	 * @param taille les dimensions de cette image
	 * @param deplacable si true, le PanelImage est deplacable
	 */
	public PanelImage(Image image, Dimension taille, boolean deplacable) {
		setImage(image);
		setTaille(taille);
		if(deplacable) {
			addMouseListener(this);
			addMouseMotionListener(this);
		}
	}
	
	/**
	 * Remplace ou ajoute une image dans le PanelImage
	 * @param image
	 */
	public void setImage(Image image) {
		this.image = image;
		largeur = image.getWidth(null);
		longueur = image.getHeight(null);
		repaint();
	}
	
	/**
	 * @return l'image contenue dans le PanelImage
	 */
	public Image getImage() {
		return image;
	}
	
	/**
	 * modifie l'opacite de l'image de fond
	 * @param opacite l'opacite de l'image de fond, comprise entre 0 (transparente) et 1 (opaque)
	 */
	public void setOpacite(float opacite) {
		this.opacite = opacite;
		repaint();
	}
	
	/**
	 * @return l'opacite de l'image de fond
	 */
	public float getOpacite() {
		return opacite;
	}
	
	/**
	 * Change la taille de l'image de fond du PanelImage
	 * @param largeur la nouvelle largeur de l'image de fond
	 * @param longueur la nouvelle longueur de l'image de fond
	 */
	public void setTaille(int largeur, int longueur) {
		this.largeur = largeur;
		this.longueur = longueur;
	}
	
	/**
	 * Change la taille de l'image de fond du PanelImage
	 * @param dimension les nouvelles dimensions de l'image de fond
	 */
	public void setTaille(Dimension dimension) {
		this.largeur = (int) dimension.getWidth();
		this.longueur = (int) dimension.getHeight();
	}
	
	/**
	 * @return les dimensions de l'image de fond
	 */
	public Dimension getTaille() {
		return new Dimension(largeur, longueur);
	}
	
	/**
	 * @return la largeur de l'image de fond
	 */
	public int getLargeur() {
		return largeur;
	}

	/**
	 * @return la longueur de l'image de fond
	 */
	public int getLongueur() {
		return longueur;
	}
	
	@Override
	public void paintComponent(Graphics g){
		Graphics2D g2d = (Graphics2D) g;
		g2d.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR);
		g2d.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, opacite));
		g2d.drawImage(image, 0, 0, largeur, longueur, null);
	}

	@Override
	public void mouseClicked(MouseEvent e) {

	}

	@Override
	public void mouseEntered(MouseEvent e) {

	}

	@Override
	public void mouseExited(MouseEvent e) {

	}

	@Override
	public void mousePressed(MouseEvent e) {
		x=(int) (e.getXOnScreen()-getLocation().getX());
		y=(int) (e.getYOnScreen()-getLocation().getY());
	}

	@Override
	public void mouseReleased(MouseEvent e) {

	}

	@Override
	public void mouseDragged(MouseEvent e) {
		Point initial = getLocation();
		int nouveauX = e.getXOnScreen()-x;
		int nouveauY = e.getYOnScreen()-y;
		setLocation(nouveauX, nouveauY);
		if(getLocation().getX()<0) setLocation(0,(int)getLocation().getY());
		if(getLocation().getX()>super.getSize().getWidth()) setLocation((int)(super.getSize().getWidth()),(int)getLocation().getY());
		if(getLocation().getY()<0) setLocation((int)getLocation().getX(),0);
		if(getLocation().getY()>super.getSize().getHeight()) setLocation((int)getLocation().getX(), (int)(super.getSize().getHeight()));
		if(!getLocation().equals(initial)) repaint();
	}

	@Override
	public void mouseMoved(MouseEvent e) {

	}
}
